<?php

namespace Mnv\Modules\Telegram;

use Exception;
use Mnv\Core\DB;

/**
 * Class TelegramUser
 * @package Mnv\Modules\Telegram
 */
class TelegramUser
{
    private string $_table_telegram_user = 'telegram_user';
    private string $_table_telegram_user_chat = 'telegram_user_chat';
    private string $_table_telegram_user_params = 'telegram_user_params';

    /** @var int|null $userId */
    private ?int $userId = null;

    /** @var int $chatId */
    private $chatId;

    /** @var int $messageId */
    private $messageId;

    /** @var mixed|null  */
    private $phone_number;

    /**
     * TelegramUser constructor.
     * @param $data
     * @throws Exception
     */
    public function __construct($data)
    {
//        sendMessage(json_encode($data,JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT));

        $this->chatId           = $data['chat']['id'];
        $this->messageId        = $data['message_id'];

        if (isset($data['contact']['phone_number'])) {
            $this->phone_number = $data['contact']['phone_number'];
        } else if (!empty($phone = $this->getPhoneNumber())) {
            $this->phone_number = $phone;
        } else {
            $this->phone_number = null;
        }

        if (!$this->isUserSet()) {
            $this->createUser($data);
        } else {
            $this->updateUser($data);
        }
    }

    /** @param $page int */
    public function setPage(int $page) {
        $this->updateParams(['page' => $page]);
    }
    public function getPage(){
        return $this->getParam('page');
    }

    /** @param $step int */
    public function setStep(int $step) {
        $this->updateParams(['step' => $step]);
    }
    public function getStep(){
        return $this->getParam('step');
    }

    /** @param string $lang */
    public function setLanguage(string $lang){
        $this->updateParams(['lang' => $lang]);
    }
    public function getLanguage() {
        return $this->getParam('lang');
    }

    /** @param string $district */
    public function setDistrict(string $district) {
        $this->updateParams(['district' => $district]);
    }
    public function getDistrict() {
        return $this->getParam('district');
    }

    /** @param int $phoneNumber */
    public function setPhoneNumber(int $phoneNumber) {
        $this->updateParams(['phone' => $phoneNumber]);
    }
    public function getPhoneNumber() {
        return $this->getParam('phone');
    }

    /** @param int $category */
    public function setCategoryId(int $category) {
        $this->updateParams(['categoryId' => $category]);
    }
    public function getCategoryId() {
        return $this->getParam('categoryId');
    }

    /** @param int|null $articleId */
    public function setArticleId(int $articleId) {
        $this->updateParams(['articleId' => $articleId]);
    }
    public function getArticleId() {
        return $this->getParam('articleId');
    }

    public function setLatitude($latitude) {
        $this->updateParams(['latitude' => $latitude]);
    }
    public function getLatitude() {
        return $this->getParam('latitude');
    }
    public function setLongitude($longitude) {
        $this->updateParams(['longitude' => $longitude]);
    }
    public function getLongitude() {
        return $this->getParam('longitude');
    }

    /** @return bool */
    private function isUserSet(): bool
    {
        $this->userId = DB::init()->connect()->table($this->_table_telegram_user_chat)->select('user_id')->where('chat_id', $this->chatId)->getValue();
//        sendMessage($this->userId);
        if (!is_null($this->userId)) {
            return true;
        }

        return false;
    }

    /** Создание и получение user_id */
    private function createUserId() {
        $this->userId = DB::init()->connect()->table($this->_table_telegram_user_chat)->insert(['chat_id' => $this->chatId]);
    }

    /** Создание нового пользователя
     * @throws Exception
     */
    private function createUser(array $data)
    {
        if (!empty($this->chatId)) {
            $this->createUserId();
            $this->setParams();
            $params = [
                'id'            => $this->userId,
                'contact'       => $this->phone_number,
                'is_bot'        => $data['from']['is_bot'] ?? false,
                'is_premium'    => $data['from']['is_premium'] ?? false,
                'firstName'     => $data['chat']['first_name'] ?? null,
                'lastName'      => $data['chat']['last_name'] ?? null,
                'username'      => $data['chat']['username'] ?? null,
                'language_code' => $data['from']['language_code'] ?? null,
                'createdAt'     => date('Y-m-d H:i:s', $data['date']),
                'mailing'       => 'no'
            ];

            DB::init()->connect()->table($this->_table_telegram_user)->insert($params);
//            sendMessage(json_encode(DB::init()->connect()->getQuery(),JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT));

        } else {
            throw new Exception('Empty chat_id');
        }
    }

    /** Обновление пользователя
     * @throws Exception
     */
    private function updateUser(array $data)
    {
        if ($this->userId) {
            $params = [
                'contact'       => $this->phone_number,
//                'is_bot'        => $data['from']['is_bot'] ?? false,
//                'is_premium'    => $data['from']['is_premium'] ?? false,
                'firstName'     => $data['chat']['first_name'] ?? null,
                'lastName'      => $data['chat']['last_name'] ?? null,
                'username'      => $data['chat']['username'] ?? null,
                'language_code' => $data['from']['language_code'] ?? null,
                'updatedAt'     => date('Y-m-d H:i:s', $data['date'])
            ];
            DB::init()->connect()->table($this->_table_telegram_user)->where('id', $this->userId)->update($params);

        } else {
            throw new Exception('Empty user_id');
        }
    }


    /**
     */
    private function setParams()
    {
        $params = ['id' => $this->userId, 'message_id' => $this->messageId, 'page' => 0, 'lang' => 'ru', 'step' => 0];
        DB::init()->connect()->table($this->_table_telegram_user_params)->where('id', $this->userId)->replace($params);
    }

    /** @param array $params */
    private function updateParams(array $params)
    {
        $params = array_merge(['id' => $this->userId, 'message_id' =>  $this->messageId], $params);
        DB::init()->connect()->table($this->_table_telegram_user_params)->where('id', $this->userId)->update($params);
    }


    /**
     * @param $param
     * @return mixed|string|null
     */
    private function getParam($param)
    {
//        sendMessage(DB::init()->connect()->getQuery());
        return DB::init()->connect()->table($this->_table_telegram_user_params)->join($this->_table_telegram_user_chat, 'id', '=', 'user_id')
            ->select($param)->where('chat_id', $this->chatId)->getValue();
    }

    /**
     * @param $text
     * @return bool
     */
    public function addReview($text): bool
    {
//        $newText = preg_replace('%[^a-zа-я\d]%i', ' ', $text);
        $params = [
            'user_id'       => $this->userId,
            'message_id'    => $this->messageId,
            'review'        => htmlspecialchars($text),
            'createdAt'     => date('Y-m-d H:i:s', time()),
        ];
        if (DB::init()->connect()->table('telegram_user_review')->insert($params)) return true;

        return false;
    }

}