<?php


namespace Mnv\Modules\Telegram\Pages;


use Mnv\Core\DB;
use Mnv\Core\Files\Image\ImageSizes;
use Mnv\Modules\Telegram\Commands\Commands;
use Mnv\Modules\Telegram\Helpers\PointCoordinates;
use Mnv\Modules\Telegram\Pagination\Exceptions\InlineKeyboardPaginationException;
use Mnv\Modules\Telegram\Pagination\InlineKeyboardPagination;
use Mnv\Modules\Telegram\Telegram;
use Mnv\Modules\Telegram\TelegramUser;
use Mnv\Modules\Telegram\Yandex\Exception\CurlError;
use Mnv\Modules\Telegram\Yandex\Exception\MapsError;
use Mnv\Modules\Telegram\Yandex\Exception\ServerError;
use Mnv\Modules\Telegram\Yandex\Exception\YandexException;
use Mnv\Modules\Telegram\Yandex\YandexApi;

class Articles
{
    /** @var int $perPage */
    private int $perPage = 10;

    /** @var array $objects */
    public array $objects = [];
    private int $selectedPage = 1;


    /** @var string $usage */
    protected string $usage = '/Articles';
    protected string $usage_category = '/Category';

    /** @var $object */
    public $object;

    /** @var null $instance */
    private static $instance = null;

    /** @return Articles|null */
    public static function init(): ?Articles
    {
        if (null === static::$instance) {
            static::$instance = new Articles();
        }

        return static::$instance;
    }


    public function execute(Telegram $telegram, TelegramUser $user, int $chatID)
    {

        if ($telegram->Callback_Query()) {
            $MessageID = $telegram->MessageID();
            $callbackData = $telegram->Callback_Data();

//            sendMessage(json_encode($callbackData, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT));
            // Разбираем команду на части, отбрасывая первую - мы уже в так уже в нужном классе
            $data = array_slice(explode('::', $callbackData), 1);

            $method = array_shift($data);

            if (isset($data[2])) $this->selectedPage = $data[2];

            $response = $method && method_exists($this, $method) ? $this->$method($telegram, $user, $data[0]) : $this->objects($telegram, $user, $data[0]);

            if (is_array($response)) {
                $response['chat_id'] = $chatID;
                $response['message_id'] = $MessageID;
                $response['parse_mode'] = "HTML";

                return isset($response['hide']) ? $telegram->deleteMessage(['chat_id' => $chatID, 'message_id' => $MessageID]) : $telegram->editMessageText($response);
            }

            return $telegram->sendMessage([
                'chat_id'       => $chatID,
                'message_id'    => $MessageID,
                'parse_mode'    => "HTML",
                'text'          => Commands::getMessageText('mistake'),
            ]);
        }

        return $telegram->sendMessage(array_merge([
            'chat_id' => $chatID,
            'parse_mode' => "HTML"
        ], $this->objects($telegram, $user, 2)));
    }


    /**
     * Страница с объектами
     *
     * @param Telegram $telegram
     * @param TelegramUser $user
     * @param int $sectionId
     * @return array
     */
    public function objects(Telegram $telegram, TelegramUser $user, int $sectionId): array
    {

        $district = $user->getDistrict();
        $latitude = $user->getLatitude();
        $longitude = $user->getLongitude();
//
        // TODO: реализовать поиск по координатам (радиус поиска)
        /** поиск по координатам */
        if (!empty($latitude) && !empty($longitude)) {
            $this->getNearestLocation($sectionId, $latitude, $longitude);
        } /** поиск по району */
        else if (!empty($district)) {
            $this->getObjects($sectionId, $district);
        } /** поиск только по категориям */
        else {
            $this->getObjects($sectionId, null);
        }

        try {
            $contentObjects = '';

            $ikp = new InlineKeyboardPagination($this->objects, $this->usage, 'objects', $sectionId, $this->selectedPage, $this->perPage);
            $ikp->setMaxButtons(5, true);
            $ikp->setLabels(['default' => '%d', 'first' => '« %d', 'previous' => '%d', 'current' => '· %d ·', 'next' => '%d', 'last' => '%d »']);
            $ikp->setCallbackDataFormat('{COMMAND}::{METHOD}::{ID}::{OLD_PAGE}::{NEW_PAGE}');
            $objects = $ikp->getPagination($this->selectedPage);

            $i = ($this->selectedPage == 1) ? 1 : ($this->selectedPage - 1) . 1;
            foreach ($objects['items'] as $object) {
                if (!empty($object->phone)) {
                    if (!empty($object->distance)) {
                        $contentObjects .= "$i. <b>$object->title</b> \n📍 $object->address \n☎️ $object->phone \n🕐 $object->workingHours \nРасстояние " . $object->distance . "  км \n️🔗 Подробнее: /card_$object->articleId \n\n";
                    } else {
                        $contentObjects .= "$i. <b>$object->title</b> \n📍 $object->address \n☎️ $object->phone \n🕐 $object->workingHours \n️🔗 Подробнее: /card_$object->articleId \n\n";
                    }
                } else {
                    if (!empty($object->distance)) {
                        $contentObjects .= "$i. <b>$object->title</b> \n📍 $object->address \n🕐 $object->workingHours \nРасстояние " . $object->distance . " км  \n🔗️ Подробнее: /card_$object->articleId \n\n";
                    } else {
                        $contentObjects .= "$i. <b>$object->title</b> \n📍 $object->address \n🕐 $object->workingHours \n🔗️ Подробнее: /card_$object->articleId \n\n";
                    }
                }
                $i++;
            }
//            sendMessage(count($objects['keyboard']));
            if (count($objects['keyboard']) > 1) $keyboardButton = [$objects['keyboard']];
            $keyboardButton[] = [$telegram->buildInlineKeyboardButton(Commands::getMessageText('back_to_category'), null, implode('::', [$this->usage_category, 'subCategories', $this->getParentId($sectionId)]))];
            return [
                'text'          => $contentObjects,
                'reply_markup'  => $telegram->buildInlineKeyBoard($keyboardButton),
            ];

        } catch (InlineKeyboardPaginationException $e) {
            return [
                'text' => $e->getMessage(),
            ];

        }
    }


    /**
     * Получение объектов по району и категориям
     *
     * @param int $sectionId
     * @param string|null $district
     */
    private function getObjects(int $sectionId, ?string $district): void
    {
        if (!empty($district)) DB::init()->connect()->like('address', "%$district%");
        $this->objects = DB::init()->connect()->table('articles')->select('articleId, sectionId, title, fileName, address, phone, workingHours, coordinates')
            ->where('status', 'visible')->where('typeContent', 'categorys')->where('sectionId', $sectionId)
            ->orderBy('articleId DESC')->limit(10000)->getAll();

    }


    /**
     * Получение объектов по району
     *
     * @param int $sectionId
     * @param $latitude
     * @param $longitude
     */
    private function getNearestLocation(int $sectionId, $latitude, $longitude)
    {
        $a = new PointCoordinates($latitude, $longitude);
        $points = [];
        if ($pointObjects = DB::init()->connect()->table('articles')->select('articleId, coordinates')->where('status', 'visible')->where('typeContent', 'categorys')
            ->where('sectionId', $sectionId)->where('coordinates', '<>', '')->orderBy('articleId DESC')->limit(10000)->getAll()) {

            foreach ($pointObjects as $p) {
                if (!empty($p->coordinates)) {
                    $loc = explode(", ", $p->coordinates);
                    $points[$p->articleId] = new PointCoordinates($loc[0], $loc[1]);
                }
            }

            if (!empty($points)) {
                foreach ($points as $key => $point) {
                    $object = DB::init()->connect()->table('articles')->select('articleId, sectionId, title, fileName, address, phone, workingHours, coordinates')->where('articleId', $key)->orderBy('articleId DESC')->get();
                    if (!empty($object)) {
//                        $distance = $a->distanceTo($point);
                        // поиск по радиусу в 3км
                        if (($distance = $a->getDistanceBetweenPoints($latitude, $longitude, $point->x, $point->y)) < 3) {
                            $distance = number_format($distance, 2);
                            $this->objects[$distance] = (object)[
                                'articleId'     => $object->articleId,
                                'sectionId'     => $object->sectionId,
                                'title'         => $object->title,
                                'fileName'      => $object->fileName,
                                'address'       => $object->address,
                                'phone'         => $object->phone,
                                'workingHours'  => $object->workingHours,
                                'coordinates'   => $object->coordinates,
                                'distance'      => $distance
                            ];
                        }
                    }
                }
            }

            if ($this->objects) {
                ksort($this->objects);
                $this->objects = array_values($this->objects);
            }
        }

        file_put_contents('locations.txt', json_encode($this->objects, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT));
    }


    /**
     * Получить ID раздела
     *
     * @param int $sectionId
     * @return int
     */
    private function getParentId(int $sectionId): int
    {
        if ($parentId = DB::init()->connect()->table('sections')->select('parentId')->where('sectionId', $sectionId)->getValue()) return $parentId;

        return 2;
    }

    public function isObject(int $articleId): bool
    {
        if (DB::init()->connect()->table('articles')->select('articleId')->where('status', 'visible')->where('articleId', $articleId)->get()) return true;

        return false;
    }

    /**
     * Получение объекта
     *
     * @param int $articleId
     * @param Telegram $telegram
     * @param int $chatID
     * @param int|null $MessageID
     * @param bool $isEditMessage
     */
    public function object(int $articleId, Telegram $telegram, int $chatID, ?int $MessageID, bool $isEditMessage = false)
    {

        if ($this->isObject($articleId)) {

            $this->object = DB::init()->connect()->table('articles')->select('articleId, sectionId, title, fileName, address, phone, workingHours, coordinates, content, imageUrl')
                ->where('status', 'visible')->where('articleId', $articleId)->get();
            // получение картинки
            if ($this->object->imageUrl) {
                $image = $this->object->imageUrl;
            } else {
                $this->getObjectFiles($this->object->articleId);
                $image = !empty($this->object->image) ? GLOBAL_URL . $this->object->image->original : GLOBAL_URL . '/uploads/default.jpg';
            }
            $contentText = strip_tags($this->object->content);
            $contentText = trim(html_entity_decode($contentText));

            if (!empty($this->object->phone)) {
                $contentObjects = "<b>" . $this->object->title . "</b> \n📍 " . $this->object->address . " \n☎️ " . $this->object->phone . " \n<i>🕐 " . $this->object->workingHours . "</i> \n\n$contentText \n<a href='$image'>&#8205;</a>";
            } else {
                $contentObjects = "<b>" . $this->object->title . "</b> \n📍 " . $this->object->address . " \n<i>🕐 " . $this->object->workingHours . "</i> \n\n$contentText \n<a href='$image'>&#8205;</a>";
            }

            $options = [
                [
                    $telegram->buildInlineKeyboardButton(Commands::getMessageText('continue_searching'), null, implode('::', [$this->usage, 'hideObject', $this->getParentId($this->object->sectionId)])),
                    $telegram->buildInlineKeyboardButton(Commands::getMessageText('view_map'), null, "/map_" . $this->object->coordinates)
                ]
            ];
            $keyboard = $telegram->buildInlineKeyBoard($options);

            if ($isEditMessage) {
                $telegram->editMessageText([
                    'chat_id'       => $chatID,
                    'message_id'    => $MessageID,
                    'reply_markup'  => $keyboard,
                    'text'          => $contentObjects,
                    'parse_mode'    => "HTML"
                ]);
            } else {
                $telegram->sendMessage([
                    'chat_id'       => $chatID,
                    'reply_markup'  => $keyboard,
                    'text'          => $contentObjects,
                    'parse_mode'    => "HTML"
                ]);
            }
        } else {
            $telegram->sendMessage([
                'chat_id'       => $chatID,
                'text'          => Commands::getMessageText('object_not_found'),
                'parse_mode'    => "HTML"
            ]);
        }
    }

    /**
     *
     * @param Telegram $telegram
     * @param TelegramUser $user
     * @param int $sectionId
     * @return bool[]
     */
    private function hideObject(Telegram $telegram, TelegramUser $user, int $sectionId): array
    {
        return [
            'hide' => true
        ];
    }

    /**
     * Получение изображения
     *
     * @param $articleId
     */
    private function getObjectFiles($articleId)
    {

        $images = DB::init()->connect()->table('article_images')->where('articleId', $articleId)->orderBy('orderBy')->getAll();
        foreach ($images as $imageId => $image) {
            if ($file = DB::init()->connect()->table('files')->select('fileId, folder, path, fileName, size')->where('fileId', $image->fileId)->get()) {
                if ($image->type === 'general') {
                    $this->object->image = ImageSizes::init()->get($image, $file);
                } else if ($image->type === 'gallery') {
                    $this->object->gallery[$imageId] = ImageSizes::init()->get($image, $file);
                } else {
                    $this->object->docs[$imageId] = ImageSizes::init()->get($image, $file);
                }
            }
        }
    }



    public function searchYandex(Telegram $telegram, TelegramUser $user, $query, int $chatID)
    {
        $district = $user->getDistrict();
        $latitude = $user->getLatitude();
        $longitude = $user->getLongitude();

        try {
            $api = new YandexApi('6d4c034a-ab5b-494a-bf44-e45116081378', '');

            // центр области поиска
            if (!empty($latitude) && !empty($longitude)) {
                $api->setLL($longitude,$latitude);
            } else {
                $api->setLL(69.279737,41.311151);
            }

//            $this->selectedPage -= 1;
//            sendMessage($this->selectedPage );
            $response =  $api->setQuery($query)->setLimit(500)->setLang(YandexApi::LANG_RU)->load()->getResponse();
//            $response = $api->getResponse();
            $countAddress = $response->getFoundCount();
            $getQuery = $response->getQuery();
            $telegram->sendMessage([
                'chat_id'       => $telegram->chatID(),
                'text'          => "кол-во найденных адресов " . $countAddress,
                'parse_mode'    => "HTML"
            ]);

            $collection = $response->getList();

            $newCollection = [];
            foreach ($collection as $item) $newCollection[] = (object)$item->_data;

            try {
                $contentObjects = '';
//                sendMessage($this->selectedPage);
//                sendMessage($this->perPage);
                $ikp = new InlineKeyboardPagination($newCollection, '/articles', 'objects', 0, $this->selectedPage, $this->perPage);
                $ikp->setMaxButtons(5, true);
                $ikp->setLabels(['default' => '%d', 'first' => '« %d', 'previous' => '%d', 'current' => '· %d ·', 'next' => '%d', 'last' => '%d »']);
                $ikp->setCallbackDataFormat('{COMMAND}::{METHOD}::{ID}::{OLD_PAGE}::{NEW_PAGE}');
                $objects = $ikp->getPagination($this->selectedPage);

//         sendMessage(json_encode($newCollection,JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT));
                $i = 1;
                foreach ($newCollection as $object) {
                    $contentObjects .= "$i. <b>$object->title</b> \n" .
                        "📍 $object->address \n" .
                        "☎️ $object->phone\n" .
                        "🕐 $object->workingHours\n" .
                        "Категории: " . implode(", ", $object->categories) .
                        "\n🔗 Подробнее: /card_$object->id \n\n";

                    $i++;
                }
//            sendMessage(count($objects['keyboard']));
                if (count($objects['keyboard']) > 1) {
                    $keyboardButton = [$objects['keyboard']];
                    $telegram->sendMessage(['chat_id' => $telegram->chatID(), 'reply_markup' => $telegram->buildInlineKeyBoard($keyboardButton), 'text' => $contentObjects, 'parse_mode' => "HTML"]);
                } else {
                    $telegram->sendMessage(['chat_id' => $telegram->chatID(), 'text' => $contentObjects, 'parse_mode' => "HTML"]);
                }
            } catch (InlineKeyboardPaginationException $e) {
                $telegram->sendMessage(['chat_id' => $telegram->chatID(), 'text' => $e->getMessage(), 'parse_mode' => "HTML"]);
            }

//            file_put_contents('result.txt', json_encode($newCollection, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT));

        } catch (YandexException|CurlError|MapsError|ServerError $e) {
            $telegram->sendMessage([
                'chat_id'       => $telegram->chatID(),
                'text'          => json_encode($e->getMessage(),JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT),
                'parse_mode'    => "HTML"
            ]);
        }
    }

}