<?php

namespace Mnv\Models\Users;

use Mnv\Core\DB;
use Mnv\Core\Managers\Role;
use Mnv\Models\Users\Exceptions\NoUserException;
use Mnv\Models\Users\Exceptions\EmptyUserIdException;
use Mnv\Models\Users\Exceptions\NotUpdateUserException;
use Mnv\Models\Users\Exceptions\NotUserApproveException;
use Mnv\Models\Users\Exceptions\UserNotDeletedException;
use Mnv\Models\Users\Exceptions\CannotBeRemovedDeveloperException;
use Mnv\Models\Users\Exceptions\CannotBeRemovedAdministratorException;

class SiteUser extends AbstractUser implements UserInterface
{
    private static $_table = 'users';

    /** UsersAdmin constructor */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Получение всех пользователей
     *
     * @param string|null $query (optional)
     * @param string|null $status (optional)
     * @param int $page
     * @param int $limit
     * @param null $columnSortOrder (optional)
     * @param null $columnName (optional)
     * @param $manager
     */
    public function getAll(?string $query, ?string $status, int $page, int $limit, $columnSortOrder = null, $columnName = null, $manager) : void
    {
        $columnIndex = $columnSortOrder[0]['column'];

        if ($columnSortOrder) {
            $orderBy = $columnName[$columnIndex]['data'] . " " . strtoupper($columnSortOrder[0]['dir']);
        } else {
            $orderBy = 'userId DESC';
        }
        $this->getUsers($query, $status, $page, $limit, 'siteUser', $orderBy, $manager);
        foreach ($this->users as $userId => $user) {
            $this->users[$userId]->statusName = lang('users:statuses:'.$user->status);
            $this->users[$userId]->registered = langDate(adjustTime(date('m/d/Y H:i:s', $user->registered), false, 'd.m.Y'));
        }
    }

    /**
     * получение кол-во пользователей
     *
     * @param string|null $query
     * @param string|null $status
     * @param $manager
     */
    public function total(?string $query, ?string $status, $manager) : void
    {
        $this->getCountUsers($query, $status, 'siteUser', $manager);
    }

    /**
     * Получить кол-во пользователей при фильтрации
     *
     * @param string|null $query
     * @param $manager
     */
    public function countTableFiltered(?string $query, $manager) : void
    {
        $this->countTableFilteredUsers($query, 'siteUser', $manager);
    }

    /**
     * Редактирование пользователя
     *
     * @throws NoUserException
     */
    public function edit()
    {
        if (!empty($this->userId)) {
            $this->getUser();
            if ($this->user !== null) {
                $this->user->image = $this->getFile($this->user->fileId);
            } else {
                throw new NoUserException();
            }
        }
    }

    /**
     * TODO сделано
     * Сохранение нового пользователя
     *
     * @param $user
     * @param $manager
     * @return bool
     */
    public function add($user, $manager): bool
    {
        $user->password       = \password_hash($user->password, \PASSWORD_DEFAULT);
        $user->registered     = \time();
        $user->accessLevel    = Role::CONSUMER;

        if ($this->userId = DB::init()->connect()->table(static::$_table)->insert((array)$user)) return true;

        return false;
    }

    /**
     * TODO сделано
     * Обновление информации пользователя
     * @param $user
     * @param $manager
     * @throws NotUpdateUserException
     */
    public function update($user, $manager)
    {
        $user->password   = !empty($user->password) ? \password_hash($user->password, \PASSWORD_DEFAULT) : $this->getUserOldPassword($this->userId);
        $user->modifiedBy = $manager->userId;
        $user->modifiedOn = gmdate('Y-m-d H:i:s');
        $user->accessLevel = Role::CONSUMER;

        if ($this->updateUser((array)$user)) {
            $this->getUser();

            $this->user->image = $this->getFile($this->user->fileId);
        } else {
            throw new NotUpdateUserException();
        }
    }

    /**
     * Удаление пользователя
     *
     *
     * @throws NoUserException
     * @throws EmptyUserIdException
     * @throws UserNotDeletedException
     * @throws CannotBeRemovedDeveloperException
     * @throws CannotBeRemovedAdministratorException
     */
    public function remove()
    {
        if ( $this->userId !== null ) {
            $this->getUser();
            if ($this->user !== null) {
                if ( $this->user->accessLevel == Role::DEVELOPER ) {
                    if ( !$this->getUserByRole(Role::DEVELOPER) ) {
                        throw new CannotBeRemovedDeveloperException();
                    }
                } else if ( $this->user->accessLevel == Role::ADMIN ) {
                    if ( !$this->getUserByRole(Role::ADMIN) ) {
                        throw new CannotBeRemovedAdministratorException();
                    }
                }
                if ( !$this->removeUser() ) {
                    throw new UserNotDeletedException();
                }
            } else {
                throw new NoUserException();
            }
        } else {
            throw new EmptyUserIdException();
        }

    }

    /**
     * Обновление статуса
     *
     * @throws NoUserException
     * @throws EmptyUserIdException
     * @throws NotUserApproveException
     */
    public function approve()
    {
        if ( $this->userId !== null ) {
            $this->getUser();
            if ($this->user !== null) {
                if (!$this->approveUser()) {
                    throw new NotUserApproveException();
                }
             }else {
                throw new NoUserException();
            }
        } else {
            throw new EmptyUserIdException();
        }

    }

}