<?php

namespace Mnv\Models\Contents;

use Mnv\Core\DB;
use Mnv\Core\Mnv;
use Mnv\Core\Managers\Role;
use Mnv\Core\Files\Image\ImageSizes;

/**
 * Class AbstractProduct
 * @package Mnv\Models\Contents
 */
class AbstractProduct extends Mnv
{

    private static $_table              = 'articles';
    private static $_table_images       = 'article_images';
    private static $_files              = 'files';
    private static $_table_comments     = 'comments';
    private static $_table_variants     = 'product_variants';
    private $_fields = array('articleId', 'title', 'alias', 'sectionId', 'isFeatured', 'url', 'publishedOn', 'orderBy', 'typeContent', 'status' , 'enableGallery');

    /**
     * весь контент
     * @var $contents
     */
    public $contents;

    /**
     * кол-во контента
     * @var $total
     */
    public $total = 0;

    /**
     * контент
     * @var Content
     */
    public $content;

//    public $oldContent;

    /**
     * контент ID
     * @var $articleId
     */
    public $articleId;


    public $result;

    public $image;

    public $imageId;

    /**
     * фильтрация / сортировка контента
     *
     * @param string|null $query
     * @param string|null $status
     * @param string|null $typeContent
     * @param array|null $sectionIds
     */
    private function sorting(?string $query, ?string $status, ?string $typeContent, ?array $sectionIds, $manager)
    {
        /** проверка уровня доступа и получение того контента который добавил пользователь со статусом `ПИСАТЕЛЬ` */
        if ($manager->accessLevel == Role::WRITER) DB::init()->connect()->where('addedBy', $manager->userId);

        if (!empty($query))        DB::init()->connect()->like('title', "%$query%")->orLike('content', "%$query%");
        if (!empty($status))       DB::init()->connect()->where('status',  $status);
        if (!empty($typeContent))  DB::init()->connect()->where('typeContent', $typeContent);
        if (!empty($sectionIds))   DB::init()->connect()->in('sectionId', $sectionIds);
    }

    /**
     * @param string|null $query
     * @param string|null $status
     * @param string|null $typeContent
     * @param array|null $sectionIds
     */
    protected function getContents(?string $query, ?string $status, ?string $typeContent, ?array $sectionIds, $manager)
    {
        /** фильтрация / сортировка */
        $this->sorting($query, $status, $typeContent, $sectionIds, $manager);

        $this->contents = DB::init()->connect()->table(self::$_table)
            ->select($this->_fields)
            ->orderBy('orderBy', 'DESC')
            ->indexKey('articleId')
            ->pagination($this->limit, $this->page)
            ->getAllIndexes();
//         print_r(DB::init()->connect()->getQuery());
    }

    /**
     * Получить кол-во контента
     *
     * @param string|null $query
     * @param string|null $status
     * @param string|null $typeContent
     * @param array|null $sectionIds
     * @param $manager
     */
    protected function getCountContents(?string $query, ?string $status, ?string $typeContent, ?array $sectionIds, $manager) : void
    {
        /** фильтрация / сортировка */
        $this->sorting($query, $status, $typeContent, $sectionIds, $manager);

        $this->total = DB::init()->connect()->table(self::$_table)->count('*', 'count')->getValue();
    }

    /**
     * Получить контент
     */
    protected function getContent() : void
    {
        $this->content = DB::init()->connect()->table(self::$_table)->where('articleId',  $this->articleId)->get();
    }

    /**
     * Удаление контента
     */
    protected function removeContent($manager): bool
    {
        /** проверка уровня доступа и получение того контента который добавил пользователь со статусом `ПИСАТЕЛЬ` */
        if ($manager->accessLevel == Role::WRITER) DB::init()->connect()->where('addedBy', $manager->userId);

        /** удаление контента и удаление записей из базы данных прикрепленных картинок и комментариев к этому контенту */
        if (DB::init()->connect()->table(self::$_table)->where('articleId', $this->articleId)->delete()) {
            DB::init()->connect()->table(self::$_table_comments)->where('articleId', $this->articleId)->delete();
            DB::init()->connect()->table(self::$_table_images)->where('articleId', $this->articleId)->delete();

            if (DB::init()->connect()->table(self::$_table_variants)->where('productId', $this->articleId)->get()) {
                DB::init()->connect()->table(self::$_table_variants)->where('productId', $this->articleId)->delete();
            }

            if (DB::init()->connect()->table('product_options')->where('productId', $this->articleId)->get()) {
                DB::init()->connect()->table('product_options')->where('productId', $this->articleId)->delete();
            }

            return true;
        }

        return false;
    }


    protected function insertContent(array $content): bool
    {
        if ($this->articleId = DB::init()->connect()->table(self::$_table)->insert($content)) return true;

        return false;
    }

    protected function updateContent(array $content): bool
    {
        if (DB::init()->connect()->table(self::$_table)->where('articleId',$this->articleId)->update($content)) return true;

        return false;
    }



    /** категории */
    public function getCategories($sectionIds)
    {
        $categoriesKeys = array_flip($sectionIds);
        $categories = array();
        $categoriesArr = DB::init()->connect()->table('sections')->select('parentId')->in('sectionId', array_keys($categoriesKeys))->groupBy('parentId')->getAll();
        foreach ($categoriesArr as $item) {
            $categories[] = $item->parentId;
        }
        return $categories;
    }

    /**
     * Проверка на совпадение и получение fileName
     *
     * @param string|null $fileName
     * @return int|mixed|string|null
     */
    public function checkFileName(?string $fileName)
    {
        $fileName = strtolower($fileName);
        if (empty($fileName)) {
            $maxId = DB::init()->connect()->table(self::$_table)->max('articleId')->getValue();
            return $maxId ? $maxId + 1 : 1;
        } elseif (empty($this->articleId)) {
            if ($fileName = DB::init()->connect()->table(self::$_table)->select('fileName')->where('LOWER(fileName)','=', $fileName)->getValue()) return $fileName;
        } else {
            if ($fileName = DB::init()->connect()->table(self::$_table)->select('fileName')->where('articleId','<>', $this->articleId)->where('LOWER(fileName)','=', $fileName)->getValue()) return $fileName;
        }
        return null;
    }

    /**
     * Утверждение статьи
     *
     * @param $manager
     * @return bool
     */
    protected function approveContent($manager)
    {
        /* проверка уровня доступа */
        if ($manager->accessLevel == Role::WRITER) DB::init()->connect()->where('addedBy','=', $manager->userId);

        if (DB::init()->connect()->table(self::$_table)->where('articleId',$this->articleId)->update(['status' => 'visible'])) return true;

        return false;
    }

    /**
     * Обновление статуса контента
     *
     * @param $contentUpdate
     * @return bool
     */
    protected function statusContent($contentUpdate)
    {
        if (DB::init()->connect()->table(self::$_table)->where('articleId', $this->articleId)->update($contentUpdate)) return true;

        return false;
    }

    /**
     * Сортировка контента
     *
     * @param $ids
     * @param $articleIds
     */
    public function reorder($ids, $articleIds)
    {
        foreach ($articleIds as $i => $articleId) {
            //print_r( $ids[$i] . '-' . $articleId . ' : ');
            DB::init()->connect()->table(self::$_table)->where('articleId', $ids[$i])->update(['orderBy' => $articleId]);
        }
    }

    /**
     * получать информацию о пользователях, которые создали и изменили статью
     */
    protected function getManagerInfo()
    {
        if (isset($this->content->addedBy) && isset($this->content->modifiedBy)) {
            $managers = DB::init()->connect()->table('users')->select('userId, loginName')->in('userId', [$this->content->addedBy, $this->content->modifiedBy])->indexKey('userId')->getAllIndexes();
            $this->content->addedBy = empty($managers[$this->content->addedBy]) ? unknownUser() : $managers[$this->content->addedBy];
            $this->content->modifiedBy = empty($managers[$this->content->modifiedBy]) ? unknownUser() : $managers[$this->content->modifiedBy];
            $this->content->addedOn = adjustTime($this->content->addedOn, false, 'd.m.Y H:i');
            $this->content->modifiedOn = adjustTime($this->content->modifiedOn, false, 'd.m.Y H:i');
            $this->content->publishedOn = adjustTime($this->content->publishedOn);
        }
    }


    /**
     * Удаление основной картинки
     * @param $imageId
     * @return bool
     */
    public function deleteImage($imageId): bool
    {
        if (!empty($imageId)) {
            DB::init()->connect()->table(self::$_table_images)->where('imageId', '=', $imageId)->delete();
            return true;
        }

        return false;
    }



    /**
     * Добавление или замена информации у картинки в галереи
     *
     * @param $imageId
     * @param $imageAlt
     * @param $imageLink
     * @return array|false|int
     */
    public function editPictureInfo($imageId, $imageAlt, $imageLink)
    {

        if (DB::init()->connect()->table(self::$_table_images)->where('imageId','=',$imageId)->update(['alt' => $imageAlt, 'link' => $imageLink])) {
            return DB::init()->connect()->table(self::$_table_images)->where('imageId','=', $imageId)->get();
        }
        return false;
    }

    /**
     * Сортировка картинок в галереи
     * @param $imageIds
     * @return bool
     */
    public function sortPictures($imageIds)
    {
        $order = 1;
        foreach ($imageIds as  $imageId) {
            DB::init()->connect()->table(self::$_table_images)->where('imageId','=',$imageId)->update(['orderBy' => $order]);
            $order++;
        }
        return true;
    }

    /**
     * Удаление картинки из галереи
     * @param $imageId
     * @return bool
     */
    public function deletePicture($imageId)
    {
        if (DB::init()->connect()->table(self::$_table_images)->where('imageId', '=', $imageId)->delete()) return true;

        return false;
    }

    /**
     * Добавление основной картинки к контенту
     *
     * @param $articleId
     * @param $images
     * @return bool|int
     */
    protected function general($articleId, $images)
    {
        $imageUpload = array_filter(['articleId' => $articleId, 'fileId' => $images['fileId'], 'type' => 'general', 'orderBy' => 0]);

        if ($imageId = DB::init()->connect()->table(self::$_table_images)->select('imageId')->where('type', '=', 'general')->where('articleId', '=', $articleId)->getValue()) {
            $imageUpload['imageId'] = $imageId;
        }

        if (DB::init()->connect()->table(self::$_table_images)->replace($imageUpload)) return true;

        return false;
    }

    /**
     * Загрузка картинок в галерею
     *
     * @param $articleId
     * @param $images
     * @return bool|int
     */
    protected function picture($articleId, $images)
    {
        $idx = 0;
        if ($order = DB::init()->connect()->table(self::$_table_images)->max('orderBy')->where('articleId', '=', $articleId)->getValue()) $idx = $order;

        foreach ($images as $fileId) {
            $idx++;
            $imageUpload = array_filter(['articleId' => $articleId, 'fileId' => $fileId, 'type' => 'gallery', 'orderBy' => $idx]);
            DB::init()->connect()->table(self::$_table_images)->replace($imageUpload);
        }
    }

    



    /**
     * получать основной картинки
     *
     * @param $articleId
     * @return ImageSizes|null
     */
    protected function getFile($articleId)
    {
        $image = null;
        $articleImage = DB::init()->connect()->table('article_images')->select('fileId')->where('type', 'general')->where('articleId', $articleId)->get();
        if ($articleImage) {
            if ($file = DB::init()->connect()->table(self::$_files)->select('*')->where('fileId', $articleImage->fileId)->get())
                $image = ImageSizes::init()->get($articleImage, $file);
        }

        return $image;
    }


    /**
     * получать основной картинки
     */

    protected function getFiles(): void
    {
        if (!empty($this->articleId)) {
            if ($images = DB::init()->connect()->table('article_images')->select('*')->where('articleId', $this->articleId)->indexKey('imageId')->orderBy('orderBy')->getAllIndexes()) {
                foreach ($images as $imageId => $image) {
                    if ($file = DB::init()->connect()->table(self::$_files)->where('fileId', $image->fileId)->get()) {
                        if ($image->type === 'general') {
                            $this->content->image = ImageSizes::init()->get($image, $file);

                        } elseif ($image->type === 'gallery') {
                            $this->content->gallery[$imageId] = ImageSizes::init()->get($image, $file);
                        } else {
                            $this->content->docs[$imageId] = ImageSizes::init()->get($image, $file);
                        }
                    }
                }
            }
        }

    }



    /**
     * @param $content
     * @param array $column
     * @return mixed
     */
    protected function unsetColumns(&$content, array $column)
    {
        foreach ($content as $key => $value) {
            if (in_array($key, $column)) {
                unset($content[$key]);
            }
        }
        return $content;
    }
}