<?php

namespace Mnv\Core\Files\Image;

use Imagine\Image\ImagineInterface;
use Mnv\Core\DB;
use Mnv\Core\Mnv;
use Imagine\Image\Box;
use Imagine\Image\ImageInterface;
use Mnv\Core\Files\Image\Exceptions\InvalidParamException;

use League\Flysystem\Filesystem;
use League\Flysystem\FilesystemException;

/**
 * Class ImageGenerator
 * @package Mnv\Core\Imagine\Action
 */
class ImageGenerator extends Mnv implements ActionInterface
{
    /** @var string $_files */
    private $_files = 'files';

    /** @var string[] $sizes */
    private $sizes = array('large', 'medium', 'small');

    /** @var  Filesystem */
    private $filesystem;

    /** @var $imagine */
    private $imagineDriver;

    /** @var $file */
    private $file;

    /** @var false|string $codeImage */
    private $codeImage;

    /** @var string $destination_directory */
    private $destination_directory;

    /** @var string $noDomainFile */
    private $path;

    /** @var ImageInterface $image */
    private $image;
    /** @var array|false[]|int[]  */
    private $options;

    /** @var int */
    private $managerId;

    /** @var string $tmp */
    private $tmp;

    private $driver = 'gd';
    private $extension;
    private $fontWatermark = GLOBAL_ROOT . '/admin/assets/fonts/verdana.ttf';

    /** @var array $fontOptions */
    private $fontOptions;

    private $mimeType;
    private $fileSize;

    public function __construct(Filesystem $filesystem, $managerId)
    {
        parent::__construct();
        $this->managerId    = $managerId;
        $this->filesystem  = $filesystem;
    }

    /**
     *
     * @param $file
     * @param int $fileSize
     * @param string|null $mimeType
     * @return $this
     */
    public function init($file, int $fileSize, ?string $mimeType): ImageGenerator
    {

        $this->file         = $file;
        $this->mimeType     = $mimeType;
        $this->fileSize     = $fileSize;
        $root_directory     = "/uploads/";

        $driver = $this->setDriver();

        Image::setImagine($driver);

        /** если включена конвертация файла в `webp` то нам нужно знать, если это `gif` то нельзя конвертировать в `webp`, для остальных расширений разрешена конвертация в `webp` */
        if ($this->file['extension'] == 'gif') {
            $this->extension = $file['extension'];
        } else {
            $this->extension = ($this->config['force_webp'] == 1) ? "webp" : $this->file['extension'];
        }

        /** сбор опций обработки `watermark` $this->fontOptions = [
         * 'Размер шрифта (size)'        => 'watermark_font',
         * 'Цвет шрифта (color)'         => 'watermarkColor',
         * 'Прозрачность шрифта (alpha)' => 'watermark_opacity',  // пока не работает
         * 'Поворот текста (angle)'      => 'watermark_rotate'
         * ];
         */
        $this->fontOptions = [
            'size'      => $this->config['watermark_font'],
            'color'     => [
                'light' => $this->config['watermark_color_light'],
                'dark'  => $this->config['watermark_color_dark']
            ],
            'alpha'     => intval($this->config['watermark_opacity']),
            'angle'     => $this->config['watermark_rotate'],
            'margin'    => 20,
        ];

        try {
            $this->options = Image::mergeOptions($this->extension, $this->config['thumbnail_quality']);
        } catch (InvalidParamException $e) {
            print $e->getMessage();
        }

        /** название файла */
        if ($this->config['keep_original_image_name'])
            $this->codeImage = pregGetValue('/([^\/]+)\.[^\.]+$/', $this->file['name']);
        if (empty($this->codeImage))
            $this->codeImage = substr(md5(number_format(time() * rand(),0,'','')),3,12);

        /** директории */
        $this->path = $root_directory . $this->file['directory'];
        $this->destination_directory = UPLOAD_ROOT . $this->file['directory'];

        /** получение (ранее загруженного) временного файла ( по окончанию удаляем его [ $this->filesystem->delete($this->file['name']) ] ) */
        $this->tmp = $this->destination_directory . $this->file['name'];

        /** достаем временный файл для обработки */
        try {
            $this->image = Image::open($this->tmp);
        } catch (InvalidParamException $e) {
            print $e->getMessage();
        }

        return $this;
    }

    /**
     * @return bool|int|string|null
     * @throws FilesystemException
     */
    public function apply()
    {
        $images = [
            'folder'    => $this->getFolder($this->file['directory']),
            'path'      => $this->path,
            'name'      => $this->codeImage . '.' . $this->extension,
            'size'      => $this->fileSize ?? null,
            'type'      => $this->mimeType,
        ];

        $this->processingAndSavingDimensions($this->image->getSize()->getWidth(),$this->image->getSize()->getHeight(), $this->destination_directory . '/' . $this->codeImage . '.' . $this->extension);
        foreach ($this->sizes as $size) {
            $sizes = $this->autoSize($this->config[$size . '_thumb']);
            $this->filesystem->createDirectory($size);
            $this->processingAndSavingDimensions($sizes[0], $sizes[1], $this->destination_directory . '/' . $size . '/' . $this->codeImage . '.' . $this->extension);
        }

        $this->filesystem->delete($this->file['name']);


        return $this->safeFile($images, $images['folder'], $this->path);
    }

    /**
     * обработка и сохранение размеров
     *
     * @param int $width
     * @param int $height
     * @param $path
     */
    private function processingAndSavingDimensions(int $width, int $height, $path)
    {
        try {
            if ($this->config['allow_watermark'] && $this->extension != 'gif') {
                Image::textBox($this->tmp, $this->config['watermark_text'], $this->fontWatermark, $this->config['watermark_position'], $this->fontOptions)->thumbnail(new Box($width, $height))->save($path, $this->options);
//                Image::text($this->tmp, $this->config['watermark_text'], $this->fontWatermark, [ $this->tempX ,  $this->tempY ], $this->fontOptions)->thumbnail(new Box($width, $height))->save($path, $this->options);
            } else {
                Image::resize($this->tmp, $width, $height)->save($path, $this->options);
//                Image::thumbnail($this->tmp, $width, $height)->save($path, $options);
//                Image::crop($this->tmp, $width, $height)->save($path, $options);
            }
        } catch (InvalidParamException $e) {
            echo $e->getMessage();
        }
    }

    /**
     * @param $file
     * @param $folder
     * @param $path
     * @return bool|int|string|null
     */
    public function safeFile($file, $folder, $path)
    {
        if ($file) {
            $insertFiles = array_filter([
                'folder'    => $folder,
                'path'      => $path,
                'fileName'  => $file['name'],
                'size'      => $file['size'],
                'mime_type' => $file['type'],
                'addedBy'   => !empty($this->managerId) ? $this->managerId : 0,
                'addedOn'   => gmdate('Y-m-d H:i:s'),
            ]);

            return $this->saveDatabase($insertFiles);
        }

        return false;
    }

    /**
     * Сохранение в базу
     *
     * @param array|null $insertFileInfo
     * @return bool|int|string|null
     */
    private function saveDatabase(?array $insertFileInfo)
    {
        if ($insertFileInfo) {
            if ($fileId = DB::init()->connect()->table($this->_files)->insert($insertFileInfo)) return $fileId;

            return false;
        }

        return false;
    }

    /**
     * Получить драйвер обработчика файла
     */
    private function setDriver(): ImagineInterface
    {

        if ($this->config['image_driver'] != 2) {
            if (extension_loaded('imagick') && class_exists('Imagick'))	{
                $this->driver  =  'imagick';

                if (!\Imagick::queryFormats('WEBP') && function_exists('imagewebp') && $this->config['image_driver'] != 1 ) {
                    $this->driver  =  'gd';
                }
            }
        }



        if ($this->driver == 'imagick') {
            return new \Imagine\Imagick\Imagine();
        } else if ($this->driver == 'gd') {
            return new \Imagine\Gd\Imagine();
        }else if ($this->driver == 'gmagick') {
            return new \Imagine\Gmagick\Imagine();
        }
    }


    /**
     * Формируем размеры для обработки файла
     * @param $size
     * @return array|false|string[]
     */
    public function autoSize($size)
    {
        $sizes = [];
        if (strpos($size, 'x') !== false) {

            $sizes = explode("x", $size);

            if (count($sizes) == 2) {
                $sizes[0] = intval($sizes[0]);
                $sizes[1] = intval($sizes[1]);

            } else {
                $sizes[0] = intval($sizes[0]);
                $sizes[1] = intval($sizes[0]);
            }

        } else {
            $sizes[0] = intval($size);
            $sizes[1] = intval($size);
        }

        return $sizes;
    }

    /**
     * Получить из пути последнее значение (будет папкой)
     * @param $path
     * @return false|mixed|string
     */
    private function getFolder($path)
    {
        $folder = explode('/', $path);
        $folder = array_diff($folder, array(''));
        return end($folder);
    }

    public function __destruct() { }

}